import Card from './Card.js';

export default class Roulette{

    constructor(){
        this.canvasSize = 320;                                  //Taille du canvas
        this.images = ["roulette.png", "cards.png", "text.png", "popup.png"];   //Images à charger
        this.nbCards = 52;                                      //Nombre de carte dans le paquet
        this.scoreInitial = 5;                                  //Score initial du joueur
        this.duration = 2;                                      //Durée de l'animation de la révélation
        //Coordonnée et options de tous les boutons cliquables
        this.btnList = [{x:279, y:15, w:26, h:26, type:"replay", isGameOver:false},
                        {x:124, y:199, w:72, h:16, type:"replay", isGameOver:true},
                        {x:23, y:119, w:82, h:32, type:"color", value:0, multiply:2},
                        {x:215, y:119, w:82, h:32, type:"color", value:1, multiply:2},
                        {x:23, y:169, w:32, h:32, type:"sign", value:0, multiply:5},
                        {x:73, y:169, w:32, h:32, type:"sign", value:2, multiply:5},
                        {x:215, y:169, w:32, h:32, type:"sign", value:1, multiply:5},
                        {x:265, y:169, w:32, h:32, type:"sign", value:3, multiply:5}
                    ];
    }

    newParty(){                             //Se déclenche au début de la partie
        this.score = this.scoreInitial;     //Score du joueur
        this.timer = this.duration;         //Temps écoulé depuis le début de l'animation
        this.mise = 1;                      //Mise actuelle
        this.historic = [];                 //Historic des signes tirés
        this.choice = null;                 //Choix du joueur
        this.cards = [];                    //Liste des cartes du paquet
        this.createCards();                 //Création des cartes du paquet + Mélange
        this.render();                      //Rendu du canvas
    }
    createCards(){
        //On crée autant de carte que demandé
        for(let index=0; index<this.nbCards; index++){
            let card = new Card(index); card.isBig = true;
            this.cards.push(card);
        }
        //On mélange les cartes une fois qu'elles sont toutes créées
        this.cards = this.cards.sort(() => Math.random() - 0.5); 
    }
    get isGameOver(){ 
        // Retourne vrai si toutes le paquet est vide ou si le score est à 0
        return this.cards.length === 0 || this.score <= 0;  
    }
    //Retourne vrai si le joueur peut jouer car aucune animation en cours
    get ready(){ return this.timer >= this.duration; }

    updateMise(value) {     
        //Met à jour la main du joueur en fonction de la valeur donnée
        //La mise ne peut pas être inférieur à 1, ni supérieur à 9, ni supérieur au score
        this.mise = value;
        if(this.mise < 1){ this.mise = 1; }
        if(this.mise > 9){ this.mise = 9; }
        if(this.mise > this.score){ this.mise = this.score; } 
    }
    updateChoice(btn) {
        //Prise en compte du choix du joueur et lancement de l'animation de révélation
        this.timer = 0;
        this.choice = btn;
        //Launch gameloop for animation
        requestAnimationFrame(this.loopRender);
    }
    get isPrediction(){
        //Retourne vrai si la prédiction du joueur était bonne
        return this.cards[0][this.choice.type] === this.choice.value;
    }

    // Methodes de rendu
    loopRender = (timestamp) => {
        //On calcul le temps écoulé depuis le dernier rafraichissement
        const delta = APP.getDelta(timestamp);
        if(delta){
            this.timer += delta;    //On ajoute la valeur au timer de l'animation
            if(this.timer > this.duration){ this.timer = this.duration; }
            this.render();          //On rend à nouveau l'image
        }        
        //Tant que l'animation n'est pas terminé, on continue
        if(!this.ready){ requestAnimationFrame(this.loopRender); }
        //Sinon on applique le résultat des gains
        else{ this.handleEndLoop(); }
    }
    render() {
        APP.context.drawImage(APP.images['roulette'], 0, 0);    //Fond du jeu
        this.drawCards(APP.context);                            //Paquet de carte
        this.drawMise(APP.context);                             //Mise
        this.drawScore(APP.context);                            //Score
        this.drawDeckNumber(APP.context);                       //Nombre de cartes restantes
        this.drawHistoric(APP.context);                         //Historique    
        this.drawPopup(APP.context);                            //Popup en fin de partie

    }
    drawCards(ctx) {
        //Deuxième carte (face cachée)
        ctx.save();
        ctx.translate(APP.canvas.width/2, APP.canvas.height/2); 
        this.cards[1].draw(ctx);
        ctx.restore();
        //Carte dessus
        ctx.save();
        let card = this.cards[0];
        ctx.translate(APP.canvas.width/2, APP.canvas.height/2); 
        //Animation flip
        //Premier tier de l'animation, on retourne la carte
        if(!this.ready && this.timer < this.duration/3){ 
            const linear = APP.easeLinear(this.timer, (this.duration/3));
            if(this.timer<(this.duration/6)){ ctx.scale(1-linear*2, 1); }
            else{ card.isFlipped = true; ctx.scale((linear-0.5)*4, 2); }
        }
        //Tant que la carte est visible, on double sa taille
        if(!this.ready && this.timer > this.duration/3 ){ ctx.scale(2, 2); }
        //Animation résultat
        //Si la prédiction etait juste la carte disparait lentement
        //Si elle était fausse, elle fait également un mouvement de droite à gauche
        if(!this.ready && this.timer > (this.duration/3)*2){ 
            const linear = APP.easeLinear((this.timer-(this.duration/3)), (this.duration/3));
            const elastic = APP.easeOutElastic((this.timer-(this.duration/3*2)), (this.duration/3));
            ctx.globalAlpha = 1 - linear;
            if(!this.isPrediction){ ctx.rotate((45 - (45 * elastic)) * Math.PI / 180); }
        }        
        card.draw(ctx);
        ctx.restore();
    }
    drawMise(ctx){
        // Ecriture de la mise en taille x4
        let char = this.mise.toString();
        const img = APP.images['text'];
        ctx.save();
        ctx.translate(144, 24);
        ctx.scale(4,4);
        ctx.drawImage(img, parseInt(char*8), 5, 8, 12, 0, 0, 8, 12);
        ctx.restore();
        // Ajout du bandeau par dessus
        APP.context.drawImage(APP.images['roulette'], 140, 320, 64, 32, 128, 0, 64, 32);
    }
    drawDeckNumber(ctx) {
        ctx.save();
        ctx.translate(225, 252);
        this.drawText(ctx, this.cards.length, 0);
        ctx.restore();
    }
    drawScore(ctx) {
        ctx.save();
        ctx.translate(77, 22);
        this.drawText(ctx, this.score);
        ctx.restore();
    }
    drawHistoric(ctx) {
        for(let i=0; i<9; i++){
            let x = 0;
            if(this.historic.length > i){ x = 28 + (this.historic[i] * 28); }
            ctx.drawImage(APP.images['roulette'], x, 320, 28, 28, 26+(i*30), 267, 28, 28);
        }
    }
    drawPopup(ctx) {
        if(!this.isGameOver) return;          // Si la partie n'est pas terminée, on ne dessine pas la popup
        let x = 89;
        let y = 110;
        let imgY = this.score<=0 ? 106 : 0; 
        const img = APP.images["popup"];
        ctx.drawImage(img, 0, imgY, 142, 106, x, y, 142, 106);  
    }
    drawText(ctx, value, style=1){
        let text = value.toString();
        let charWidth = 8; 
        let charHeight = 12;
        let imgY = 5;
        if(style === 0){ charWidth = 5; charHeight = 5; imgY = 0; }
        let textWidth = text.length * charWidth + 1;
        let x = -textWidth;
        const img = APP.images['text'];
        for(let index in text){
            let char = text[index];
            ctx.drawImage(img, parseInt(char*charWidth), imgY, charWidth, charHeight, x + (index * charWidth - 1), 0, charWidth, charHeight);
        }
    }

    // Méthodes Evenements
    handleLoaded() {
        this.newParty();            // Quand toutes les images sont chargées, on lance une nouvelle partie
    }

    handleClick(x, y) {
        if(!this.ready) return;
        let btn = this.clicButton(x, y);
        if(!btn) return;            // Aucun bouton n'est dans la zone de clic
        if(btn.type === "replay" && btn.isGameOver === this.isGameOver){ this.newParty(); return; }
        if(this.isGameOver) return;
        this.updateChoice(btn);     // On sauvegarde le choix s'il s'agit d'un bouton de prédiction
    }

    clicButton(x, y) {
        for(let btn of this.btnList){
            // Si le clic se trouve dans la zone du bouton, on le renvoit
            if(x >= btn.x && x <= btn.x + btn.w && y >= btn.y && y <= btn.y + btn.h){ return btn; }
        }
        return false;
    }

    handleEndLoop() {
        if(this.isPrediction){  //En cas de victoire la mise * le multiplicateur sont ajouté au score
            this.score += (this.mise * this.choice.multiply); 
            this.updateMise(this.mise+1);
        }else{                  //Sinon on sousttrait la mise * le multiplicateur au score
            this.score -= (this.mise * this.choice.multiply);
            this.updateMise(1);
        }
        if(this.score<0){ this.score = 0; }         //Le score ne peut être inférieur à 0
        this.historic.unshift(this.cards[0].sign);  //Sauvegarde du résultat dans l'historique
        this.choice = null;                         //On efface le choix pour la prochaine prédiction  
        this.cards.shift();                         //On supprime la carte du dessus
        this.render();                              //On rend une dernière fois le canvas mis à jour
    }

}